﻿using NVCC.Models;
using NVCC.Repos.PatientRepository;
using NVCC.WebUI.Infrastructure;
using NVCC.WebUI.Models;
using System;
using System.Collections.Generic;
using System.Configuration;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Web.Mvc;

namespace NVCC.WebUI.Controllers
{
    [NvccAuthorize]
    [OutputCache(NoStore = true, Duration = 0, VaryByParam = "*")]
    public class PatientProfileController : Controller
    {
        private readonly IPatientProfileService _patientProfileService;
        private readonly IUserService _userService;
        private readonly IAodService _aodService;
        private readonly ICombinePdfService _combinePdfService;
        private readonly IAccessiblePdfService _accessiblePdfService;
        private readonly IImageService _imageService;

        private static readonly int MaxTotalSizeOfAttachedPdfsInMB = int.Parse(ConfigurationManager.AppSettings["MaxTotalSizeOfAttachedPdfs"]) / 1048576;
        private static readonly int MaxSizeOfAttachedPdfInMB = int.Parse(ConfigurationManager.AppSettings["MaxSizeOfAttachedPdf"]) / 1048576;
        private static readonly int MaxNumberOfAttachedPdfs = int.Parse(ConfigurationManager.AppSettings["MaxNumberOfAttachedPdfs"]);
        private static readonly string MaxTotalSizeOfPdfsErrorMessage = string.Format("The total size of all attached PDFs has exceeded the {0}MB limit", MaxTotalSizeOfAttachedPdfsInMB);
        private static readonly string MaxSizeOfIndividualPdfErrorMessage = string.Format("Attached PDFs must be {0}MB or less each.", MaxSizeOfAttachedPdfInMB);
        private static readonly string MaxNumberOfAttachedPdfsErrorMessage = string.Format("The maximum number of additional PDFs that can be attached to a RefDoc Package is {0}.", MaxNumberOfAttachedPdfs);



        public PatientProfileController(IPatientProfileService patientProfileService,
                                        IUserService userService,
                                        IAodService aodService,
                                        ICombinePdfService combinePdfService,
                                        IAccessiblePdfService accessiblePdfService, 
                                        IImageService imageService)
        {
            _patientProfileService = patientProfileService;
            _userService = userService;
            _aodService = aodService;
            _combinePdfService = combinePdfService;
            _accessiblePdfService = accessiblePdfService;
            _imageService = imageService;
        }

        [HttpGet]
        public  ActionResult Index(int? patientSid, bool includeVistAImaging)
        {
            if (patientSid == null)
                return RedirectToAction("Index", "Home");

            var patient = HttpContextManager.Current.Session!=null ? 
                          HttpContextManager.Current.Session[patientSid.ToString()] as Patient: null;
            if (patient == null)
                return RedirectToAction("Index", "Home");

            if(patient.PatientSid!=patientSid)
                return RedirectToAction("Index", "Home");

            User user = _userService.GetUser();
            ViaUser viaUser = _userService.GetViaUser(patient.Station);

            if (viaUser == null)
            {
                TempData["InfoMessage"] = "Unable to identifer user.";
                return RedirectToAction("Login", "User");
            }

            var stopwatch = new Stopwatch();
            stopwatch.Start();
            PatientProfile patientProfile;
            try
            {
                patientProfile = _patientProfileService.GetPatientProfile(patient, viaUser);
                patientProfile.UserInfo = user;
                              
                if (patientProfile.VistaUser.Fault && patientProfile.VistaUser.FaultMessage != null)
                {
                    string viaFaultMessage=null;

                    if (patientProfile.VistaUser.FaultMessage.Contains("The queryBean.provider.userId value has expired")
                        ||
                        patientProfile.VistaUser.FaultMessage.Contains("The queryBean.provider.userId value is not an encrypted valid value")
                        ||
                        patientProfile.VistaUser.FaultMessage.Contains("queryBean.provider.userId cannot be null"))
                        viaFaultMessage = "Your User Session Expired. Please submit credentials again.";
                    else if(patientProfile.VistaUser.FaultMessage.Contains("Division specified is invalid for user"))
                    {
                        TempData["InfoMessage"] = "Division specified is invalid for user";
                        return RedirectToAction("Division", "User", new { station = patient.Station });
                    }
                    else if (patientProfile.VistaUser.FaultMessage.Contains("Division") && patientProfile.VistaUser.FaultMessage.Contains("not supported"))
                    {
                        TempData["InfoMessage"] = "Selected division is not supported for this station";
                        return RedirectToAction("Division", "User", new { station = patient.Station });
                    }
                    else if (patientProfile.VistaUser.FaultMessage.Contains("InstitutionMappingNotFoundException"))
                    {
                        TempData["InfoMessage"] = "Selected divison does not correspond to any recognized station. Please select a different one or submit a work order to have your division in that station's VistA instance updated to match the station you are in.";
                        return RedirectToAction("Division", "User", new { station = patient.Station });
                    }
                    else
                        viaFaultMessage = patientProfile.VistaUser.FaultMessage;
                    TempData["InfoMessage"] = viaFaultMessage;
                    return RedirectToAction("Login", "User");
                }
            }
          
            catch (PatientTimeoutException)
            {
                return View("Timeout",
                    new PatientTimeoutViewModel {
                        PatientSid = patientSid,
                        PatientName = patient.PatientName
                    });
            }
            catch (Exception)
            {
                throw;
            }
            
            stopwatch.Stop();
            patientProfile.Patient = patient;

            HttpContextManager.Current.Session[patientProfile.Patient.PatientIcn] = patientProfile;
            var patientProfileViewModel = new PatientProfileViewModel(patientProfile);
            patientProfileViewModel.IncludeVistAImaging = includeVistAImaging;

            string id = patientProfileViewModel.cacheInSession();
            TempData["ElapsedTime"] = ViewBag.ElapsedTime;

            _patientProfileService.LogItem(stopwatch, patient, user.DomainPlusNetworkUserName, "PatientProfile");

            return RedirectToAction("Select", new { id });
        }

        [HttpPost]
        public ActionResult Timeout([Bind(Include = "PatientSid, PatientName")] PatientTimeoutViewModel patient)
        {
            return View(patient);
        }

        [HttpGet]
        public ActionResult Select(string id)
        {
            PatientProfileViewModel patientProfileViewModel = PatientProfileViewModel.retrieveFromSession(id);
            if (patientProfileViewModel == null)
            {
                return RedirectToAction("Index", "Home");
            }
            ViewBag.ElapsedTime = TempData["ElapsedTime"];
            ViewBag.PdfErrorMessage = TempData["PdfErroMessage"];

            //Load VistA Imaging metadata for this patient
            
            if(patientProfileViewModel.IncludeVistAImaging 
               && patientProfileViewModel.PatientProfile.UserInfo.CurrentDefaultFacility > 0 
               && !String.IsNullOrEmpty(patientProfileViewModel.PatientIcn)
               && patientProfileViewModel.PatientProfile.ImagingStudiesResultSet == null)
            {              
                    _imageService.GetImagesMetaData(patientProfileViewModel);
                    if (patientProfileViewModel.PatientProfile.ImagingStudiesResultSet != null && patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study.Count>0)
                        patientProfileViewModel.VistAImagingStudiesSelected = patientProfileViewModel.VistAImagingStudiesSelected == null ? Enumerable.Repeat(false, patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study.Count()).ToList() : patientProfileViewModel.VistAImagingStudiesSelected;      
            }


            if (patientProfileViewModel.DisclosureInfo == null)
            {
                patientProfileViewModel.DisclosureInfo = new Disclosure();
            }
            if (patientProfileViewModel.DisclosureInfo.AdditionalObjects == null)
            {
                patientProfileViewModel.DisclosureInfo.AdditionalObjects = new List<AdditionalObject>();
                patientProfileViewModel.DisclosureInfo.AdditionalObjects.Add(new AdditionalObject { ObjectDateString = "", ObjectType = "", OtherTypeText = "" });
                patientProfileViewModel.DisclosureInfo.AdditionalObjects.Add(new AdditionalObject { ObjectDateString = "", ObjectType = "", OtherTypeText = "" });
                patientProfileViewModel.DisclosureInfo.AdditionalObjects.Add(new AdditionalObject { ObjectDateString = "", ObjectType = "", OtherTypeText = "" });
                patientProfileViewModel.DisclosureInfo.AdditionalObjects.Add(new AdditionalObject { ObjectDateString = "", ObjectType = "", OtherTypeText = "" });
            }

            ViewBag.IncludeVistAImaging = patientProfileViewModel.IncludeVistAImaging;
            return View("Index", patientProfileViewModel);
        }

        /*ValidateInput(false) fixes a bug that caused app to crash when 
         * html-like bracket tags were inserted into text*/
        [HttpPost]
        [ValidateInput(false)]
        [ValidateAntiForgeryToken]
        public ActionResult Display([Bind(Include = "PatientIcn, ReportOptions,VistAImagingStudiesSelected, AppointmentsSelected, AuthorizationSelected, ConsultSelected, ProblemDiagnosesSelected, RadiologyReportsSelected, AdditionalLabs,ImagingStudiesResultSet, ProgressNotes, AdditionalRecords, AdditionalAuthorizations, AdditionalConsults, AdditionalRadiology, AdditionalOrders, ReferralType, NotesSelected, PdfsToAttach, AttachedPdfs, DisclosureInfo, AdditionalObject, AdditionalObjects, IncludeVistAImaging")] PatientProfileViewModel patientProfileViewModel)
        {
            // The PatientProfile component of the patientProfileViewModel is not passed back 
            // from the form, so restore it to the view model by reloading it from the Session.
            if (!patientProfileViewModel.RefreshPatientProfile())
            {
                TempData["Message"] = "Unable to retrieve cached patient data. Please search for patient again.";
                return RedirectToAction("Error", "Home");
            }

            //Remove additional objects with Removed flag equals true, if they exist in the list
            if (patientProfileViewModel.DisclosureInfo.AdditionalObjects!=null && patientProfileViewModel.DisclosureInfo.AdditionalObjects.Any(item => item.Removed))
            {
                IList<AdditionalObject> objects = new List<AdditionalObject>();
                foreach (AdditionalObject obj in patientProfileViewModel.DisclosureInfo.AdditionalObjects)
                {
                    if (!obj.Removed)
                    {
                        objects.Add(obj);
                    }
                }
                patientProfileViewModel.DisclosureInfo.AdditionalObjects = objects;
            }

            if (patientProfileViewModel.IncludeVistAImaging && patientProfileViewModel.VistAImagingStudiesSelected != null && patientProfileViewModel.VistAImagingStudiesSelected.Any(v => v))
            {
                // Retrieve image meta data for this patient      
                _imageService.GetImagesMetaData(patientProfileViewModel, true);
                int vistaImagesSelected = (patientProfileViewModel.VistAImagingStudiesSelected != null
                    ? patientProfileViewModel.VistAImagingStudiesSelected.Count(selected => selected)
                    : 0);

                if (vistaImagesSelected > 0)
                {
                    _imageService.SetupVistAImages(patientProfileViewModel);
                }
            }

            // if refresh is false, there is an error and patientProfileViewModel is not consistent; need to indicate this somehow and to do something about it.
                        // This is one place that a PatientProfileViewModel is created from scratch and it is cached with a handle (ID) to be able to get it back


            const string nonPdfErrorMessage = "Only PDFs can be attached to a RefDoc Package";
            const string pdfCorruptedErrorMessage = "One or more attached PDFs were corrupted and have been removed.";

            patientProfileViewModel.RemoveSelectedAttachedPdfs();

            var pdfsToAttach = patientProfileViewModel.PdfsToAttach;

            var id = patientProfileViewModel.cacheInSession();

            if (pdfsToAttach != null && pdfsToAttach.Any())
            {
                pdfsToAttach.RemoveAt(pdfsToAttach.Count() - 1);
                if (!patientProfileViewModel.AttachmentsArePdfs(pdfsToAttach))
                {
                    TempData["PdfErrorMessage"] = nonPdfErrorMessage;
                    return RedirectToAction("Select", new { id });
                }
                if (!patientProfileViewModel.SizeOfEachPdfIsWithinLimit(pdfsToAttach))
                {
                    TempData["PdfErrorMessage"] = MaxSizeOfIndividualPdfErrorMessage;
                    return RedirectToAction("Select", new { id });
                }
                if (!patientProfileViewModel.TotalSizeOfAttachedPdfsIsWithinLimit(pdfsToAttach))
                {
                    TempData["PdfErrorMessage"] = MaxTotalSizeOfPdfsErrorMessage;
                    return RedirectToAction("Select", new { id });
                }
                if (!patientProfileViewModel.NumberOfPdfsIsWithinLimit(pdfsToAttach))
                {
                    TempData["PdfErrorMessage"] = MaxNumberOfAttachedPdfsErrorMessage;
                    return RedirectToAction("Select", new { id });
                }
                TempData["PdfErrorMessage"] = null;
            }

            patientProfileViewModel.AttachPdfs(pdfsToAttach);

            if (!patientProfileViewModel.AttachedPdfsAreReadable())
            {
                TempData["PdfErrorMessage"] = pdfCorruptedErrorMessage;
                patientProfileViewModel.RemoveSelectedAttachedPdfs();
                return RedirectToAction("Select", new { id });
            }

            if (patientProfileViewModel?.DisclosureInfo?.AdditionalObjects != null && patientProfileViewModel.DisclosureInfo.AdditionalObjects.Any())
            {
                foreach (var additionalObject in patientProfileViewModel.DisclosureInfo.AdditionalObjects)
                {
                    try
                    {
                        Convert.ToDateTime(additionalObject.ObjectDateString);
                    }
                    catch
                    {
                        TempData["AodErrorMessage"] = "All additional object dates must be valid dates in MM/DD/YYYY format";
                        return RedirectToAction("Select", new { id });
                    }
                }
            }

            return RedirectToAction("Display", new { id });
        }

        [HttpGet]
        public ActionResult Display(string id)
        {
            PatientProfileViewModel patientProfileViewModel = PatientProfileViewModel.retrieveFromSession(id);
            if (patientProfileViewModel == null)
            {
                return RedirectToAction("Index", "Home");
            }            

            return View(patientProfileViewModel);
        }

        private static readonly bool DeleteUploadedPdfs = Convert.ToBoolean(ConfigurationManager.AppSettings["DeleteUploadedPdfs"]);

        [HttpGet]
        public ActionResult Pdf(String id)
        {
            User user = _userService.GetUser();
            PatientProfileViewModel patientProfileViewModel = PatientProfileViewModel.retrieveFromSession(id);
            if (patientProfileViewModel == null)
            {
                return RedirectToAction("Index", "Home");
            }
            
            //On generate PDF save the AOD information.
            patientProfileViewModel.DisclosureInfo.UserId = user.DomainPlusNetworkUserName;
            patientProfileViewModel.DisclosureInfo.Sta3n = user.CurrentDefaultFacility;
            if (user.Facilities.ContainsKey(user.CurrentDefaultFacility))
            {
                patientProfileViewModel.DisclosureInfo.VISN = user.Facilities[user.CurrentDefaultFacility].VISN;
                patientProfileViewModel.DisclosureInfo.StationName = user.Facilities[user.CurrentDefaultFacility].StationName;
            }

            _aodService.RecordAodInfo(patientProfileViewModel);

            Stopwatch stopwatch = new Stopwatch();
            stopwatch.Start();
            var stationInfo = _userService.GetStationInfo(patientProfileViewModel.PatientProfile.Patient.Station);
            patientProfileViewModel.StationInfo = stationInfo;
            stopwatch.Stop();
            _patientProfileService.LogItem(stopwatch, patientProfileViewModel.PatientProfile.Patient, user.DomainPlusNetworkUserName, "PDF");

            var refDocFileName = patientProfileViewModel.PatientProfile.Patient.PatientName + "-" + patientProfileViewModel.PatientProfile.Patient.PatientSsn.Substring(5, 4) + ".pdf";

            var html = this.RenderView("PDF", patientProfileViewModel);
            
            var leftFooter = patientProfileViewModel.PatientProfile.Patient.PatientName + " DOB: " + String.Format("{0:MM/dd/yyyy}", patientProfileViewModel.PatientProfile.Patient.DateOfBirth);

            var refDocTempPath = _accessiblePdfService.createPdf(html, leftFooter);

            var vistAImagingPdfStudiesSelected = new List<VistAImagingStudy>();

            if (patientProfileViewModel.IncludeVistAImaging &&
                patientProfileViewModel.VistAImagingStudiesSelected != null &&
                patientProfileViewModel.VistAImagingStudiesSelected.Any(s => s) &&
                patientProfileViewModel.PatientProfile.VistAImagingStudies != null &&
                patientProfileViewModel.PatientProfile.VistAImagingStudies.Any(s =>
                    s.Files != null && s.Files.Any(f => f.FileType.Contains("pdf"))))
            {
                for (int i = 0; i < patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study.Count; i++)
                {
                    if (patientProfileViewModel.VistAImagingStudiesSelected[i])
                    {
                        var study = patientProfileViewModel.PatientProfile.VistAImagingStudies.FirstOrDefault(s => s.StudyId == patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].StudyId);
                        if (study != null && study.Files.Any(f => f.FileType.Contains("pdf")))
                        {
                            vistAImagingPdfStudiesSelected.Add(study);
                        }
                    }
                }
            }

            if ((patientProfileViewModel.AttachedPdfs == null || !patientProfileViewModel.AttachedPdfs.Any()) 
                && !vistAImagingPdfStudiesSelected.Any())
            {
                return File(System.IO.File.ReadAllBytes(refDocTempPath), "application/pdf", refDocFileName);
            }

            var combinedPdfPath = _combinePdfService.CombinePdfs(refDocTempPath, 
                patientProfileViewModel.AttachedPdfs, 
                vistAImagingPdfStudiesSelected);
            
            var combinedPdf = File(System.IO.File.ReadAllBytes(combinedPdfPath), "application/pdf", refDocFileName);

            if (DeleteUploadedPdfs)
            {
                var allTempPaths = new List<string>() { combinedPdfPath, refDocTempPath };
                if (patientProfileViewModel.AttachedPdfs != null && patientProfileViewModel.AttachedPdfs.Any(a => !string.IsNullOrWhiteSpace(a.FullPath)))
                {
                    allTempPaths.AddRange(patientProfileViewModel.AttachedPdfs.Select(p => p.FullPath));
                }
                ThreadPool.QueueUserWorkItem(o => DeleteFiles(allTempPaths));
            }

            return combinedPdf;
        }

        private static readonly TimeSpan DelayDeleteTime = TimeSpan.FromMinutes(double.Parse(ConfigurationManager.AppSettings["MinutesToDelayDeletingTempPdfs"]));

        private static async void DeleteFiles(IEnumerable<string> paths)
        {
            ///We delay deleting of the temp files because the user may click to download the PDF
            ///more than once.
            await Task.Delay(DelayDeleteTime).ContinueWith(_ =>
            {
                foreach (var path in paths)
                {
                    try
                    {
                        System.IO.File.Delete(path);
                    }
                    catch
                    {
                        ///Not sure what to do in this catch block. Basically,
                        ///I want execution to continue even if Delete fails.
                    }
                }
            });
        }
    }
}
